/**
 * \file avl.h
 * \brief Fichier d'entête qui propose des avl génériques en C
 * \author N. Delestre
 * \version 1.0
 * \date 2024
 *
 */
#ifndef __AVL__
#define __AVL__
#include <stdbool.h>
#include <errno.h>
#include "arbreBinaire.h"
#include "copieLiberationComparaison.h"

/* Partie privee */
typedef AB_ArbreBinaire AVL_ArbreBinaireRecherche;

/* Partie publique */
/**
 * \def AFDI_ERREUR_MEMOIRE
 * \brief valeur que prend errno lorsqu'il n'y a plus de mémoire système
 */
#define AVL_ERREUR_MEMOIRE 1

/**
 * \fn AVL_ArbreBinaireRecherche AVL_arbreBinaireRecherche()
 * \brief Fonction qui crée un arbre binaire de recherche vide
 * \return un arbre binaire de recherche vide
 */
AVL_ArbreBinaireRecherche AVL_arbreBinaireRecherche();

/**
 * \fn bool AVL_estVide(AVL_ArbreBinaireRecherche abr)
 * \brief Fonction qui teste si un arbre binaire de recherche est vide
 * \param abr un arbre binaire de recherche
 * \return vrai si l'arbre est vide, faux sinon
 */
bool AVL_estVide(AVL_ArbreBinaireRecherche abr);

/**
 * \fn void* AVL_obtenirElement(AVL_ArbreBinaireRecherche abr)
 * \brief Fonction qui renvoie l'élément présent dans la racine d'un arbre binaire de recherche
 * \param abr un arbre binaire de recherche
 * \return l'élément de l'arbre
 * \pre l'arbre ne doit pas être vide
 */
void* AVL_obtenirElement(AVL_ArbreBinaireRecherche abr);

/**
 * \fn AVL_ArbreBinaireRecherche AVL_obtenirFilsGauche(AVL_ArbreBinaireRecherche abr)
 * \brief Fonction qui renvoie le fils gauche d'un arbre binaire de recherche
 * \param abr un arbre binaire de recherche
 * \return le fils gauche de l'arbre
 * \pre l'arbre ne doit pas être vide
 */
AVL_ArbreBinaireRecherche AVL_obtenirFilsGauche(AVL_ArbreBinaireRecherche abr);

/**
 * \fn AVL_ArbreBinaireRecherche AVL_obtenirFilsDroit(AVL_ArbreBinaireRecherche abr)
 * \brief Fonction qui renvoie le fils droit d'un arbre binaire de recherche
 * \param abr un arbre binaire de recherche
 * \return le fils droit de l'arbre
 * \pre l'arbre ne doit pas être vide
 */
AVL_ArbreBinaireRecherche AVL_obtenirFilsDroit(AVL_ArbreBinaireRecherche abr);

/**
 * \fn void AVL_estPresent(AVL_ArbreBinaireRecherche abr, void *pElement, CLC_FonctionComparer comparer)
 * \brief Fonction qui teste si un élément est présent dans un arbre binaire de recherche
 * \param abr un arbre binaire de recherche
 * \param pElement un élément
 * \param comparer une fonction de comparaison entre éléments
 * \return vrai si l'élément est présent, faux sinon
 */
bool AVL_estPresent(AVL_ArbreBinaireRecherche abr, void *pElement, CLC_FonctionComparer comparer);

/**
 * \fn void AVL_inserer(AVL_ArbreBinaireRecherche *pAbr, void *pElement, CLC_FonctionCopier copier, CLC_FonctionComparer comparer)
 * \brief Fonction qui insère un élément dans un arbre binaire de recherche
 * \param pAbr un pointeur sur un arbre binaire de recherche
 * \param pElement un élément
 * \param copier une fonction de copie d'élément
 * \param comparer une fonction de comparaison entre éléments
 * \pre l'élément n'est pas présent dans l'arbre
 */
void AVL_inserer(AVL_ArbreBinaireRecherche *pAbr, void *pElement, CLC_FonctionCopier copier, CLC_FonctionComparer comparer);

/**
 * \fn void AVL_supprimer(AVL_ArbreBinaireRecherche *pAbr, void *pElement, CLC_FonctionComparer comparer, CLC_FonctionLiberer liberer)
 * \brief Fonction qui supprime un élément d'un arbre binaire de recherche
 * \param pAbr un pointeur sur un arbre binaire de recherche
 * \param pElement un élément
 * \param copier une fonction de copie d'élément
 * \param comparer une fonction de comparaison entre éléments
 * \param liberer une fonction de libération d'élément
 */
void AVL_supprimer(AVL_ArbreBinaireRecherche *pAbr, void *pElement, CLC_FonctionCopier copier, CLC_FonctionComparer comparer, CLC_FonctionLiberer liberer);

/**
 * \fn void AVL_detruire(AVL_ArbreBinaireRecherche *pAbr, CLC_FonctionLiberer liberer)
 * \brief Fonction qui détruit un arbre binaire de recherche
 * \param pAbr un pointeur sur un arbre binaire de recherche
 * \param liberer une fonction de libération d'élément
 */
void AVL_detruire(AVL_ArbreBinaireRecherche *pAbr, CLC_FonctionLiberer liberer);
#endif
