/**
 * \file arbreBinaire.h
 * \brief Fichier d'entête qui propose des arbres binaires génériques en C
 * \author N. Delestre
 * \version 1.0
 * \date 2024
 *
 */


#ifndef __ARBRE_BINAIRE__
#define __ARBRE_BINAIRE__
#include <errno.h>
#include <stdbool.h>
#include "copieLiberationComparaison.h"

/* Partie privee */
typedef struct AB_Noeud* AB_ArbreBinaire;
typedef struct AB_Noeud {
  void* lElement;
  AB_ArbreBinaire filsGauche;
  AB_ArbreBinaire filsDroit;
}  AB_Noeud;

/* Partie publique */
/**
 * \def AB_ERREUR_MEMOIRE
 * \brief valeur que prend errno lorsqu'il n'y a plus de mémoire système
 */
#define AB_ERREUR_MEMOIRE 1

/**
 * \fn AB_ArbreBinaire AB_arbreBinaire()
 * \brief Fonction de création d'un arbre binaire vide
 *
 * \return AB_ArbreBinaire
 */
AB_ArbreBinaire AB_arbreBinaire();

/**
 * \fn int AB_estVide(AB_ArbreBinaire l)
 * \brief Fonction qui permet de savoir si un arbre est vide
 *
 * \param l l'arbre binaire
 * \return int
 */
bool AB_estVide(AB_ArbreBinaire);

/**
 * \fn AB_creerRacine(AB_ArbreBinaire fg,AB_ArbreBinaire fd,void* pElement,CLC_FonctionCopier copierElement)
 * \brief Fonction d'ajout d'un élément comme racine
 *        errno contient AB_ERREUR_MEMOIRE s'il n'y a plus assez de mémoire système
 * \attention la donnée ajoutée à la collection est dupliquée
 * \post errno=AB_ERREUR_MEMOIRE si pas assez de mémoire
 *
 * \param pfg le fils gauche
 * \param pfd le fils droit
 * \param pElement un pointeur vers l'émélemnt à ajouter
 * \param copierElement la fonction qui permet de dupliquer l'élément
 * \return la nouvelle racine
 */
AB_ArbreBinaire AB_creerRacine(AB_ArbreBinaire,AB_ArbreBinaire,void*,CLC_FonctionCopier);

/**
 * \fn AB_obtenirElement(AB_ArbreBinaire a)
 * \brief Fonction qui retourne l'élément qui se trouve à la racine de l'arbre
 * \pre l'arbre doit être non vide 
 *
 * \param a
 * \return void*
 */
void* AB_obtenirElement(AB_ArbreBinaire); 

/**
 * \fn AB_obtenirFilsGauche(AB_ArbreBinaire a)
 * \brief Fonction qui retourne le fils gauche de l'arbre
 * \pre l'arbre doit être non vide 
 *
 * \param a
 * \return AB_ArbreBinaire
 */
AB_ArbreBinaire AB_obtenirFilsGauche(AB_ArbreBinaire); 

/**
 * \fn AB_obtenirFilsDroit(AB_ArbreBinaire a)
 * \brief Fonction qui retourne le fils droit de l'arbre
 * \pre l'arbre doit être non vide 
 *
 * \param a
 * \return AB_ArbreBinaire
 */
AB_ArbreBinaire AB_obtenirFilsDroit(AB_ArbreBinaire);

/**
 * \fn AB_fixerFilsGauche(AB_ArbreBinaire *pa, AB_ArbreBinaire fg)
 * \brief Fonction qui permet de modifier le fils gauche de l'arbre
 * \pre l'arbre doit être non vide 
 *
 * \param pa pointeur vers l'arbre binaire
 * \param fg le fils gauche
 */
void AB_fixerFilsGauche(AB_ArbreBinaire*, AB_ArbreBinaire); 

/**
 * \fn AB_fixerFilsDroit(AB_ArbreBinaire *pa, AB_ArbreBinaire fd)
 * \brief Fonction qui permet de modifier le fils droit de l'arbre
 * \pre l'arbre doit être non vide 
 *
 * \param pa pointeur vers l'arbre binaire
 * \param fd le fils droit
 */
void AB_fixerFilsDroit(AB_ArbreBinaire*, AB_ArbreBinaire);

/**
 * \fn AB_fixerElement(AB_ArbreBinaire *pa, void* pElement, CLC_FonctionCopier copierElement,CLC_FonctionLiberer libererElement)
 * \brief Fonction qui permet de remplacer l'élément qui se trouve à la racine de l'arbre
 * \attention la donnée ajoutée à remplacer est supprimée de la mémoire et la donnée de remplacement est dupliquée
 * \pre l'arbre doit être non vide 
 *
 * \param pa pointeur vers l'arbre binaire
 * \param pElement un pointeur vers l'élément à mettre
 * \param copierElement la fonction qui permet de dupliquer l'élément
 * \param libererElement la fonction qui permet de supprimer l'élément déjà présent
 */
void AB_fixerElement(AB_ArbreBinaire*,void*,CLC_FonctionCopier,CLC_FonctionLiberer);

/**
 * \fn AB_supprimerRacine(AB_ArbreBinaire *pa,AB_ArbreBinaire *pfg,AB_ArbreBinaire *pfd,CLC_FonctionLiberer libererElement)
 * \brief Fonction qui permet de supprimer la racine de l'arbre
 * \pre l'arbre doit être non vide 
 * \attention la donnée qui se trouvait en tête de liste est supprimée
 *
 * \param pa pointeur vers l'arbre binaire
 * \param pfg qui permet de récupérer le fils gauche
 * \param pfd qui permet de récupérer le fils droit
 * \param libererElement la fonction qui permet de supprimer l'élément en tête de liste
 */
void AB_supprimerRacine(AB_ArbreBinaire*,AB_ArbreBinaire*,AB_ArbreBinaire*,CLC_FonctionLiberer);

/**
 * \fn AB_supprimer(AB_ArbreBinaire *pa,CLC_FonctionLiberer libererElement)
 * \brief Fonction qui permet de supprimer un arbre en profondeur, c'est-à-dire que tous les éléments de la collection sont supprimés
 *
 * \param pa pointeur vers l'arbre binaire
 * \param libererElement la fonction qui permet de supprimer chaque élément de l'arbre
 */
void AB_supprimer(AB_ArbreBinaire*, CLC_FonctionLiberer); 

/**
 * \fn AB_copier(AB_ArbreBinaire l,CLC_FonctionCopier copierElement)
 * \brief Fonction qui permet de copier un arbre en profondeur, c'est-à-dire que tous les éléments de la collection sont dupliqués
 *
 * \param l l'arbre binaire
 * \param copierElement la fonction qui permet de dupliquer un élément
 * \param libererElement la fonction qui permet de supprimer un élément
 * \return AB_ArbreBinaire
 */
AB_ArbreBinaire AB_copier(AB_ArbreBinaire,CLC_FonctionCopier,CLC_FonctionLiberer);

/**
 * \fn AB_egales(AB_ArbreBinaire a1,AB_ArbreBinaire a2,CLC_FonctionComparer comparerElement)
 * \brief Fonction qui permet de savoir si deux arnbre sont égaux
 * (pas obligatoirement identiques)
 *
 * \param a1 le premier arbre binaire
 * \param a2 le deuxième arbre binaire
 * \param comparerElement la fonction qui permet de comparer deux éléments de l'arbre 
 * \return int
 */
int AB_egaux(AB_ArbreBinaire,AB_ArbreBinaire,CLC_FonctionComparer);
#endif 
