"""Module pour des questions à choix multiples sur les capitales des pays.
"""
from dataclasses import dataclass
from math import radians, acos, cos, sin
import random
from typing import Self
from rdflib import URIRef
from SPARQLWrapper import SPARQLWrapper, JSON
from .wikidata import QuestionReponsesFactoryWikidata
from .question_reponses import QuestionReponses

@dataclass
class PaysCapitale:
    """Classe pour un pays et sa capitale."""
    pays: str
    capitale: str
    latitude_capitale: float|None = None
    longitude_capitale: float|None = None

    def __eq__(self, autre: object) -> bool:
        if not isinstance(autre, PaysCapitale):
            return False
        return self.pays == autre.pays and self.capitale == autre.capitale

    def distance(self, autre: Self) -> float:
        if self.latitude_capitale is None or self.longitude_capitale is None or autre.latitude_capitale is None or autre.longitude_capitale is None:
            return float("inf")
        try:
            lat1 = radians(self.latitude_capitale) # https://www.askpython.com/python/examples/find-distance-between-two-geo-locations
            lon1 = radians(self.longitude_capitale)
            lat2 = radians(autre.latitude_capitale)
            lon2 = radians(autre.longitude_capitale)
            return 6371.01 * acos(sin(lat1)*sin(lat2) + cos(lat1)*cos(lat2)*cos(lon1 - lon2))
        except Exception as e:
            return float("inf")


class QuestionReponsesCapitales(QuestionReponses):
    """Classe pour une question à choix multiples sur les capitales des pays."""
    def __init__(self, pays_capitales: list[PaysCapitale], nb_reponses: int=4):
        self._pays_capitales = pays_capitales
        self._pays_capitale_choisi = random.choice(self._pays_capitales)
        self._reponses = [self._pays_capitale_choisi.capitale] +\
            self._capitales_proches(self._pays_capitale_choisi.capitale, nb_reponses-1)
        super().__init__(self._reponses, 0)

    def _capitales_proches(self, capitale_choisie: str, nb_capitales: int) -> list[str]:
        pays_capitales_distances = [(pays_capitale, pays_capitale.distance(self._pays_capitale_choisi)) for pays_capitale in self._pays_capitales]
        pays_capitales_tries = sorted(pays_capitales_distances, key=lambda x: x[1])
        res = []
        i = 1
        while len(res) < nb_capitales and i < len(pays_capitales_tries):
            capitale = pays_capitales_tries[i][0].capitale
            if capitale != capitale_choisie and capitale not in res:
                res.append(capitale)
            i += 1
        return res

    @property
    def domaine(self) -> str:
        return "Géographie"

    @property
    def question(self) -> str:
        return f"Quelle est la capitale du (de la) {self._pays_capitale_choisi.pays}?"


class QuestionReponsesFactory(QuestionReponsesFactoryWikidata):
    """Classe pour une fabrique de questions à choix multiples sur les capitales des pays."""
    def __init__(self):
        self._pays_capitales = self._recuperer_pays_capitales()

    def _recuperer_pays_capitales(self) -> list[PaysCapitale]:
        def coord_en_latit_longit(coord: str) -> tuple[float, float]:
            coord = coord[6:-1]
            return float(coord.split(" ")[0]), float(coord.split(" ")[1])

        requete = self.prefixes() + "\n" + \
            """select ?nom_pays ?nom_capitale ?coord_capitale where {
  ?pays wdt:P31 wd:Q6256.
  ?pays wdt:P36 ?capitale.
  ?capitale wdt:P625 ?coord_capitale.
  ?pays rdfs:label ?nom_pays.
  filter(langmatches(lang(?nom_pays), "fr"))
  ?capitale rdfs:label ?nom_capitale.
  filter(langmatches(lang(?nom_capitale), "fr"))
}
            """
        resultats = self.execute_requete_sparql(requete, ["nom_pays", "nom_capitale", "coord_capitale"])
        return [PaysCapitale(pays=resultat[0],
                             capitale=resultat[1],
                             latitude_capitale=coord_en_latit_longit(resultat[2])[1],
                             longitude_capitale=coord_en_latit_longit(resultat[2])[0])
                for resultat in resultats]

    def question_reponses(self) -> QuestionReponses:
        return QuestionReponsesCapitales(self._pays_capitales)
