#!/usr/bin/env python3
import argparse
import os
from llm_image_et_texte_vers_texte import interroger_llm
from mnist import evaluer

def main(url_serveur: str, nom_modele: str, cle_api: str, prompt: str, action: str, chemin: str) -> None:
    """
    Fonction principale permettant soit d'interroger le LLM avec une image unique,
    soit d'évaluer la reconnaissance des chiffres MNIST sur un ensemble d'images.

    :param url_serveur: URL du serveur LLM.
    :param nom_modele: Nom du modèle à utiliser.
    :param cle_api: Clé API pour l'accès au serveur LLM.
    :param prompt: Texte de l'instruction à envoyer au modèle.
    :param action: Action à effectuer ("interroger_llm" ou "evaluer_llm").
    :param chemin: Chemin d'un fichier image (pour "interroger_llm") ou d'un répertoire (pour "evaluer_llm").
    """
    if action == "interroger_llm":
        if not os.path.isfile(chemin):
            print(f"Erreur : '{chemin}' n'est pas un fichier valide.")
            return

        texte_recu = interroger_llm(url_serveur, nom_modele, cle_api, chemin, prompt)
        if texte_recu:
            print(f"Réponse du LLM : {texte_recu}")
        else:
            print("Erreur lors de l'interrogation du LLM.")

    elif action == "evaluer_llm":
        if not os.path.isdir(chemin):
            print(f"Erreur : '{chemin}' n'est pas un répertoire valide.")
            return

        precision = evaluer(url_serveur, nom_modele, cle_api, chemin, prompt)
        print(f"Taux de reconnaissance des chiffres : {precision:.2f}%")
    
    else:
        print("Erreur : Action invalide. Choisissez 'interroger_llm' ou 'evaluer_llm'.")

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Reconnaissance de chiffres MNIST avec un LLM.")

    parser.add_argument("url_serveur", type=str, help="URL du serveur LLM")
    parser.add_argument("nom_modele", type=str, help="Nom du modèle")
    parser.add_argument("cle_api", type=str, help="Clé API pour le serveur")
    parser.add_argument("prompt", type=str, help="Prompt envoyé au modèle")
    parser.add_argument("action", type=str, choices=["interroger_llm", "evaluer_llm"], help="Action à effectuer ('interroger_llm' pour une image ou 'evaluer_llm' pour un répertoire)")
    parser.add_argument("chemin", type=str, help="Chemin du fichier image ou du répertoire d'images")

    args = parser.parse_args()

    main(args.url_serveur, args.nom_modele, args.cle_api, args.prompt, args.action, args.chemin)
