import os
import base64
import mimetypes
import requests
from typing import Optional

def encoder_image_base64(chemin_image: str) -> Optional[str]:
    """
    Encode une image en Base64 pour l'envoyer au modèle LLM.

    :param chemin_image: Chemin du fichier image à encoder.
    :return: Chaîne encodée en Base64 au format data URI ou None en cas d'erreur.
    """
    mime_type, _ = mimetypes.guess_type(chemin_image)
    
    if mime_type is None:
        print(f"Erreur : Impossible de déterminer le type MIME pour {chemin_image}")
        return None

    try:
        with open(chemin_image, "rb") as fichier_image:
            contenu_fichier = base64.b64encode(fichier_image.read()).decode("utf-8")
            return f"data:{mime_type};base64,{contenu_fichier}\n"
    except Exception as erreur:
        print(f"Erreur lors de l'encodage de l'image {chemin_image} : {erreur}")
        return None

def interroger_llm(url_serveur: str, nom_modele: str, cle_api: str, chemin_image: str, prompt: str) -> Optional[str]:
    """
    Interroge le modèle LLM avec une image et un texte, et retourne la réponse sous forme de texte.

    :param url_serveur: URL du serveur LLM.
    :param nom_modele: Nom du modèle utilisé pour la reconnaissance.
    :param cle_api: Clé API pour l'authentification au serveur.
    :param chemin_image: Chemin du fichier image à analyser.
    :param prompt: Texte envoyé au modèle pour orienter la reconnaissance.
    :return: Réponse brute du modèle sous forme de texte, ou None en cas d'erreur.
    """
    image_base64 = encoder_image_base64(chemin_image)
    
    if image_base64 is None:
        return None

    messages = [
        {
            "role": "user",
            "content": [
                {"type": "text", "text": prompt},
                {"type": "image_url", "image_url": {"url": image_base64}}
            ]
        }
    ]

    donnees = {
        "model": nom_modele,
        "temperature": 0,  # Température forcée à 0 pour des réponses déterministes
        "messages": messages
    }

    en_tetes = {
        "Authorization": f"Bearer {cle_api}",
        "Content-Type": "application/json"
    }

    try:
        reponse = requests.post(url_serveur+"/v1/chat/completions", headers=en_tetes, json=donnees)
        reponse.raise_for_status()
        resultat = reponse.json()
        return resultat['choices'][0]['message']['content']
    except Exception as erreur:
        print(f"Erreur lors de l'interrogation du LLM avec {chemin_image} : {erreur}")
        return None
