import os
from typing import Optional
from llm_image_et_texte_vers_texte import interroger_llm

def reconnaitre_chiffre(url_serveur: str, nom_modele: str, cle_api: str, chemin_image: str, prompt: str) -> Optional[int]:
    """
    Utilise `interroger_llm_image_et_texte_vers_texte()` pour obtenir une réponse textuelle,
    puis tente d'extraire un chiffre unique.

    :param url_serveur: URL du serveur LLM.
    :param nom_modele: Nom du modèle utilisé pour la reconnaissance.
    :param cle_api: Clé API pour l'authentification au serveur.
    :param chemin_image: Chemin du fichier image à analyser.
    :param prompt: Texte envoyé au modèle pour orienter la reconnaissance.
    :return: Chiffre reconnu sous forme d'entier (0-9), ou None en cas d'échec.
    """
    texte_recu = interroger_llm(url_serveur, nom_modele, cle_api, chemin_image, prompt)

    if texte_recu and texte_recu.isdigit() and len(texte_recu) == 1:
        return int(texte_recu)  # Retourne le chiffre sous forme d'entier
    else:
        print(f"Réponse non valide pour {chemin_image} : '{texte_recu}'")
        return None

def evaluer(url_serveur: str, nom_modele: str, cle_api: str, repertoire_images: str, prompt: str) -> float:
    """
    Évalue la reconnaissance des chiffres MNIST sur un ensemble d'images stockées dans des sous-répertoires.

    :param url_serveur: URL du serveur LLM.
    :param nom_modele: Nom du modèle utilisé pour la reconnaissance.
    :param cle_api: Clé API pour l'authentification au serveur.
    :param repertoire_images: Répertoire contenant des sous-répertoires de chiffres (0-9) avec les images associées.
    :param prompt: Texte envoyé au modèle pour orienter la reconnaissance.
    :return: Pourcentage de reconnaissance correcte.
    """
    total_images = 0
    nb_corrects = 0

    if not os.path.exists(repertoire_images):
        print(f"Erreur : Le répertoire '{repertoire_images}' n'existe pas.")
        return 0.0

    # Parcours des sous-répertoires (0, 1, ..., 9)
    for chiffre_str in map(str, range(10)):  # "0" à "9"
        chemin_sous_repertoire = os.path.join(repertoire_images, chiffre_str)

        if not os.path.isdir(chemin_sous_repertoire):
            continue  # Ignore si ce n'est pas un dossier

        liste_images = [f for f in os.listdir(chemin_sous_repertoire) if f.endswith(('.png', '.jpg', '.jpeg'))]

        for image in liste_images:
            chemin_image = os.path.join(chemin_sous_repertoire, image)
            chiffre_attendu = int(chiffre_str)

            prediction = reconnaitre_chiffre(url_serveur, nom_modele, cle_api, chemin_image, prompt)

            if prediction is not None and prediction == chiffre_attendu:
                nb_corrects += 1
            else:
                print(f"Erreur : {image} -> Prédit {prediction}, Attendu {chiffre_attendu}")

            total_images += 1

    taux_reconnaissance = (nb_corrects / total_images) * 100 if total_images > 0 else 0.0
    return taux_reconnaissance
