"""Module charger de réaliser la recherche et le classement de document pertinents"""
import logging

from .analyseur import analyser_requete
from .visiteur_modele_booleen import visiter as visiter_modele_booleen
from .visiteur_modele_booleen_etendu import visiter as visiter_modele_booleen_etendu
from .representation_vectorielle_tfidf import RepresentationVectorielle

def representation_vectorielle_documents_pertinents(representation_vectorielle_corpus: RepresentationVectorielle,
                                                    documents_pertinents: set[str]) -> RepresentationVectorielle:
    """ Retourne la représentation vectorielle des documents pertinents."""
    documents = sorted(list(documents_pertinents))
    mots = representation_vectorielle_corpus.mots
    tfidf = representation_vectorielle_corpus.tfidf[[representation_vectorielle_corpus.documents.index(document)\
                                                     for document in documents], :]
    return RepresentationVectorielle(mots, documents, tfidf)

def traiter_requete(requete: str,
                    index_inverse: dict[str, set[str]],
                    representation_vectorielle: RepresentationVectorielle) -> list[tuple[float,set[str]]]:
    """ Traite une requête en utilisant l'index inversé.
        Args:
            requete (str): La requête à traiter.
            index_inverse (dict[str, set[str]]): L'index inversé.
        Returns:
            list[tuple[float,set[str]]]: La liste des documents pertinents triés par similarité.
    """
    logging.info(f"Traitement de la requête : {requete}")
    ast = analyser_requete(requete)
    logging.info("Identification des documents pertinents")
    documents_pertinents = visiter_modele_booleen(ast, index_inverse)
    if len(documents_pertinents) == 0:
        return []
    logging.info("Calcul de la similarité des documents pertinents à la requete")
    representation_vectorielle = representation_vectorielle_documents_pertinents(representation_vectorielle, documents_pertinents)
    documents_similarites = {document: visiter_modele_booleen_etendu(ast, document, representation_vectorielle) for document in documents_pertinents}
    similarites_en_ordre_decroissant = sorted(list(set(documents_similarites.values())),  reverse=True)
    logging.info("Tri des documents par similarité")
    res = []
    for sim in similarites_en_ordre_decroissant:
        temp = set()
        for document in documents_pertinents:
            if documents_similarites[document] == sim:
                temp.add(document)
        res.append((sim, temp))
    return res
