"""Module implémentant de design pattern Visteur pour visiter un AST représentant 
    une recherche booléenne et retourner l'ensemble des documents pertinents."""

from functools import singledispatch
import logging

from .analyseur import Et, Ou, Non
from .representation_ensembliste import documents_contenant_mot, et, ou, non

@singledispatch
def visiter(ast: Et, index_inverse: dict[str, set[str]]) -> set[str]:
    """ Visite de l'AST en tant que noeud Et.
        Retourne l'intersection des ensembles. 
    """
    logging.debug(f"Visite de l'AST en tant que Et : {ast}")
    return et(visiter(ast.gauche, index_inverse),
              visiter(ast.droite, index_inverse))

@visiter.register
def _(ast: Ou, index_inverse: dict[str, set[str]]) -> set[str]:
    """ Visite de l'AST en tant que noeud Ou.
        Sinon retourne l'union des ensembles.
    """
    logging.debug(f"Visite de l'AST en tant que Ou : {ast}")
    return ou(visiter(ast.gauche, index_inverse),
              visiter(ast.droite, index_inverse))
                                                        
@visiter.register
def _(ast: Non, index_inverse: dict[str, set[str]]) -> set[str]:
    """ Visite de l'AST en tant que noeud Non.
        Retourne l'ensemble car ce n'est pas au niveau de ce noeud qu'on fait la négation.
    """
    logging.debug(f"Visite de l'AST en tant que Non : {ast}")
    corpus = set.union(*index_inverse.values())
    return non(visiter(ast.expr, index_inverse), corpus)

@visiter.register
def _(ast: str, index_inverse: dict[str, set[str]]) -> set[str]:
    """ Visite de l'AST en tant que noeud Mot.
        Retourne l'ensemble des documents contenant le mot.
    """
    logging.debug(f"Visite de l'AST en tant que Mot : {ast}")
    return documents_contenant_mot(index_inverse, ast)
