"""Module charger de d'indexer un corpuse de documents Web"""
from typing import Optional
import logging
from urllib.parse import urljoin
import re
import requests
from bs4 import BeautifulSoup
import nltk

nltk.download('punkt')

def contenu_balise_a_en_url(contenu_balise_a: str,
                            url_page: str) -> Optional[str]:
    """Retourne l'URL absolue contenu dans une balise a si 
    cette dernière correspond à l'URL d'une page web
    
    Args:
        contenu_balise_a (str): Le contenu de la balise a
        url_page (str): L'URL de la page web
    Returns:
        Optional[str]: L'URL absolue de la page web si elle existe
      """

    def est_url_page_html(url: str) -> bool:
        reponse_http = requests.head(url, timeout=1.0)
        return reponse_http.headers["Content-Type"].startswith("text/html")

    url = contenu_balise_a
    if url[0] == "#":
        return None
    if not url.startswith("http"):
        url = urljoin(url_page, url)
    if est_url_page_html(url):
        return url
    return None

def mots_et_liens_d_une_page_web(url: str, nb_max_liens: int) -> tuple[list[str], list[str]]:
    """Fonction qui extrait le texte (liste de mots) et les liens hypertextes d'une page web
    Args:
        url (str): L'URL de la page web
        nb_max_liens (int): Le nombre maximum de liens à extraire
    Returns:
        tuple[list[str], list[str]]: Le texte et les liens hypertextes
     """
    logging.info(f"Debut mots_et_liens_d_une_page_web {url}")
    reponse_http = requests.get(url, timeout=1.0)
    contenu_html = reponse_http.text
    interpretation_html = BeautifulSoup(contenu_html, 'html.parser')
    texte = interpretation_html.get_text()
    texte_nettoye = re.sub(r'<.*?>', '', texte).lower()
    liens_possibles = [balise_a["href"] \
                       for balise_a in interpretation_html.find_all("a", href=True)]
    liens = []
    for lien_possible in liens_possibles:
        try:
            lien = contenu_balise_a_en_url(lien_possible, url)
            if lien:
                liens.append(lien)
                if len(liens) >= nb_max_liens:
                    break
        except:
            logging.debug(f"Probleme avec le lien {lien_possible}")
    logging.info(f"Fin mots_et_liens_d_une_page_web {url}")
    return nltk.tokenize.word_tokenize(texte_nettoye), liens

def indexer_depuis_une_url_de_depart(url_depart: str,
                                     nb_documents_a_traiter=10,
                                     domaine: str="") -> dict[str, list[str]]:
    """Fonction qui indexe des documents web à partir d'une URL de départ
    et en suivant les liens hypertextes des pages web rencontrées
    Args:
        url_depart (str): L'URL de départ
        nb_documents_a_traiter (int): Le nombre de documents à indexer
        domaine (str) : ne prendre en compte uniquement les URL de ce domaine (si "" prend en compte tout URL)
    Returns:
        dict[str, list[str]]: Un index des documents web
    """
    file_urls = [url_depart]
    urls_documents_traites = set()
    index = {}
    while len(urls_documents_traites) < nb_documents_a_traiter and file_urls:
        logging.debug(f"URLS traitées : {urls_documents_traites}")
        logging.debug(f"URLS dans la file : {file_urls}")
        url = file_urls[0]
        if not url in urls_documents_traites:
            urls_documents_traites.add(url)
            try:
                mots, nouveaux_liens = mots_et_liens_d_une_page_web(url,
                                                                    nb_documents_a_traiter - len(urls_documents_traites))
                index[url] = mots
                for nouveau_lien in nouveaux_liens:
                    if nouveau_lien not in urls_documents_traites and domaine in nouveau_lien:
                        file_urls.append(nouveau_lien)
            except:
                logging.debug(f"Pb d'indexation pour {url}")
        file_urls.pop(0)
    return index
