"""Module proposant une représentation vectorielle (TF-IDF) de l'indexation d'un corpus"""
from dataclasses import dataclass
import numpy as np

@dataclass
class RepresentationVectorielle:
    """Representation vectirielle d'un corpus"""
    mots: list[str]
    documents: list[str]
    tfidf: np.array

class DocumentNAppartenantPasAuCorpusErreur(Exception):
    """Exception correspondant à l'absent d'un document dans le corpus"""

def _tf(documents_mots, documents, mots) -> np.array:
    """Calcul de la matrice tf."""
    # A remplacer
    return np.array([])


def _idf(documents_mots, mots) -> np.array:
    """Calcul de la matrice idf"""
    # A remplacer
    return np.array([])

def _tfidf(documents_mots, documents, mots) -> np.array:
    """Calcul de la matrice tf-idf nomalisé"""
    tf = _tf(documents_mots, documents, mots)
    idf = _idf(documents_mots, mots)
    return tf * (idf / max(idf))

def index_en_representation_vectorielle_tfidf(documents_mots: dict[str, list[str]]) -> RepresentationVectorielle:
    """Calcul la représentation vectorielle (TF-IDF) d'un corpus.
    Args:
        documents_mots (dict[str, list[str]]): Le corpus.
    Returns:
        RepresentationVectorielle: La représentation vectorielle (TF-IDF).
        Les documents et les mots sont triés dans l'ordre lexicographique croissant
    """
    documents = sorted(list(documents_mots.keys()))
    mots = sorted(list(set(mot for mots in documents_mots.values() for mot in mots)))
    tfidf = _tfidf(documents_mots, documents, mots)
    return RepresentationVectorielle(mots, documents, tfidf)

def document_en_vecteur(document: str,
                        representation_vectorielle: RepresentationVectorielle) -> np.array:
    """Retourne le vecteur TF-IDF d'un document.
    Args:
        document (str): Le document.
        representation_vectorielle (RepresentationVectorielle): La représentation vectorielle (TF-IDF).
    Returns:
        np.array: Le vecteur TF-IDF du document.
    """
    if document not in representation_vectorielle.documents:
        raise DocumentNAppartenantPasAuCorpusErreur(f"Le document {document} n'appartient pas au corpus")
    return representation_vectorielle.tfidf[representation_vectorielle.documents.index(document), :]

def similarite_cosinus(document1: str,
                       document2: str,
                       representation_vectorielle: RepresentationVectorielle) -> float:
    """Retourne la similarité cosinus entre deux documents.
    Args:
        document1 (str): Le premier document.
        document2 (str): Le second document.
        representation_vectorielle (RepresentationVectorielle): La représentation vectorielle (TF-IDF).
    Returns:
        float: La similarité cosinus entre les deux documents.
    """
    vecteur1 = document_en_vecteur(document1, representation_vectorielle)
    vecteur2 = document_en_vecteur(document2, representation_vectorielle)
    return np.dot(vecteur1, vecteur2) / (np.linalg.norm(vecteur1) * np.linalg.norm(vecteur2))
