"""Analyseur syntaxique des requêtes booléennes de recherche."""
from abc import ABC, abstractmethod
import ply.lex as lex
import ply.yacc as yacc
import logging

# Définition des classes pour l'AST
class OperationBinaire(ABC):
    def __init__(self, left, right):
        self.gauche = left
        self.droite = right

    @property
    @abstractmethod
    def symbole(self):
        pass

    def __repr__(self):
        return f"{self.__class__.__name__}({repr(self.gauche)}, {repr(self.droite)})"

    def __str__(self):
        return f"({str(self.gauche)} {self.symbole} {str(self.droite)})"
    
    def __eq__(self, autre):
        return self.__class__ == autre.__class__ \
            and self.gauche == autre.gauche and self.droite == autre.droite

class Et(OperationBinaire):
    @property
    def symbole(self):
        return "&"
    
    def __init__(self, left, right):
        super().__init__(left, right)

class Ou(OperationBinaire):
    @property
    def symbole(self):
        return "|"

    def __init__(self, left, right):
        super().__init__(left, right)

class Non:
    def __init__(self, expr):
        self.expr = expr

    def __repr__(self):
        return f"{self.__class__.__name__}({repr(self.expr)})"
    
    def __eq__(self, autre):
        return self.__class__ == autre.__class__ and self.expr == autre.expr

# class Mot:
#     def __init__(self, valeur):
#         self.valeur = valeur

# Liste des tokens
tokens = (
    'MOT',
    'ET',
    'OU',
    'NON',
    'LPAREN',
    'RPAREN'
)

# Expressions régulières pour les tokens simples
t_ET = r'ET'
t_OU = r'OU'
t_NON = r'NON'
t_LPAREN = r'\('
t_RPAREN = r'\)'

# Mot-clé OR
mots_cles = {
    'ET': 'ET',
    'OU': 'OU',
    'NON': 'NON',
}

def t_MOT(t):
    r'[a-zA-Z0-9_À-ÿ]+'
    t.type = mots_cles.get(t.value.upper(), 'MOT')
    return t

def t_QUOTED_MOT(t):
    r'"[^"]*"'
    t.value = t.value[1:-1] 
    t.type = 'MOT'
    return t


def t_error(t):
    print("Caractère non reconnu :", t.value[0])
    t.lexer.skip(1)

t_ignore = ' '

# Construction du lexer
lexer = lex.lex()

# Définition de la grammaire
def p_expression_or(p):
    '''expression : terme
                  | expression OU terme'''
    if len(p) == 2:
        p[0] = p[1]
    else:
        p[0] = Ou(p[1], p[3])

def p_terme_et(p):
    '''terme : facteur
             | terme ET facteur'''
    if len(p) == 2:
        p[0] = p[1]
    else:
        p[0] = Et(p[1], p[3])

def p_facteur_non(p):
    'facteur : NON facteur'
    p[0] = Non(p[2])

def p_facteur_groupe(p):
    'facteur : LPAREN expression RPAREN'
    p[0] = p[2]

def p_facteur_mot(p):
    'facteur : MOT'
    p[0] = p[1].lower()

def p_error(p):
    print("Erreur de syntaxe")

# Construction du parser
parser = yacc.yacc()

# Fonction pour parser une requête
def analyser_requete(query):
    return parser.parse(query)

def main():
    logging.basicConfig(level=logging.INFO)
    # Exemple d'utilisation 
    expression = 'aa ET bb OU "c c" ET NON dd'    
    ast = analyser_requete(expression)
    print(ast)

if __name__ == "__main__":
    main()
