"""Script principal pour l'indexation et la recherche de pages HTML."""

import argparse
import logging
import json
import numpy as np

from moteur_de_recherche_web.robot_web import indexer_depuis_une_url_de_depart
from moteur_de_recherche_web.representation_ensembliste import index_en_index_inverse
from moteur_de_recherche_web.representation_vectorielle_tfidf import RepresentationVectorielle, \
    index_en_representation_vectorielle_tfidf
from moteur_de_recherche_web.requeteur import traiter_requete

# Fonction d'indexation
def indexer(url: str, nb_pages: int, nom_fichier_json: str, domaine: str) -> None:
    """Indexation d'un corpus (URLs d'un domaine) de nb_pages pages à partir de l'URL de départ.
    L'indexation est enregistrée dans un fichier JSON.
    """
    logging.info(f"Indexation de {nb_pages} pages à partir de l'URL : {url}")
    #index_inverse, urls_documents_traites = indexer_depuis_une_url_de_depart(url, nb_pages)
    documents_mots = indexer_depuis_une_url_de_depart(url, nb_pages)
    index_inverse = index_en_index_inverse(documents_mots)
    print(f"Nb mots indexés : {len(index_inverse)}")
    print(f"Nb URL : {len(documents_mots.keys())}")
    represenation_vectorielle = index_en_representation_vectorielle_tfidf(documents_mots)
    index_inverse_serializable = {mot: list(occurrences) for mot, occurrences in index_inverse.items()}
    representation_vectorielle_serializable = {"mots": represenation_vectorielle.mots,
                                              "documents": represenation_vectorielle.documents,
                                              "tfidf": represenation_vectorielle.tfidf.tolist()}
    # Enregistrer les données au format JSON dans le fichier spécifié
    with open(nom_fichier_json, 'w', encoding="utf-8") as fichier_json:
        json.dump({'index_inverse':index_inverse_serializable,
                   'representation_vectorielle':representation_vectorielle_serializable},
                   fichier_json, indent=2)

# Fonction de recherche
def rechercher(requete: str, nom_fichier_json: str) -> None:
    """Recherche et classement de pages HTML correspondant à une requête.
    La représentation du corpus est chargée depuis un fichier JSON."""
    logging.info(f"Recherche de la chaîne : {requete}")
    with open(nom_fichier_json, 'r', encoding="utf-8") as fichier_json:
        logging.info(f"Chargement des données depuis le fichier {nom_fichier_json}")
        donnees = json.load(fichier_json)
    index_inverse = {mot: set(occurrences) for mot, occurrences in donnees['index_inverse'].items()}
    representation_vectorielle = RepresentationVectorielle(donnees['representation_vectorielle']['mots'],
                                                          donnees['representation_vectorielle']['documents'],
                                                          np.array(donnees['representation_vectorielle']['tfidf']))
    similarites_urls = traiter_requete(requete, index_inverse, representation_vectorielle)
    nb_urls = 0
    for similarite_urls in similarites_urls:
        for url in similarite_urls[1]:
            print(f"  {url} : {similarite_urls[0]}")
            nb_urls += 1
    print(f"Nb URLs : {nb_urls}")

def main() -> None:
    """
    Fonction principale du programme.
    """
    analyseur = argparse.ArgumentParser(description="Programme d'indexation et de recherche")
    sous_analyseurs = analyseur.add_subparsers(title="Actions",
                                               dest="action",
                                               required=True,
                                               help="Action à effectuer")
    indexer_parser = sous_analyseurs.add_parser('indexer',
                                                help="Indexer des pages HTML")
    indexer_parser.add_argument('-dom', '--domaine', default='insa-rouen.fr', help="Domaine unique")
    indexer_parser.add_argument('url', help="L'URL de départ")
    indexer_parser.add_argument('nombre_pages',
                                type=int,
                                help="Le nombre de pages HTML à indexer")
    rechercher_parser = sous_analyseurs.add_parser('rechercher',
                                                   help="Rechercher une chaîne de caractères")
    rechercher_parser.add_argument('requete',
                                   help="La chaîne de caractères à rechercher")
    analyseur.add_argument('-db', default='db.json',
                           help="Nom du fichier JSON pour la base de données")
    analyseur.add_argument('-v', '--verbeux', action='store_true', help="Activer le mode verbeux")
    analyseur.add_argument('-d', '--debug', action='store_true', help="Activer le mode verbeux")

    args = analyseur.parse_args()

    if args.debug:
        logging.basicConfig(level=logging.DEBUG)
    else:
        if args.verbeux:
            logging.basicConfig(level=logging.INFO)
        else:
            logging.basicConfig(level=logging.CRITICAL)

    if args.action == 'indexer':
        indexer(args.url, args.nombre_pages, args.db, args.domaine)
    elif args.action == 'rechercher':
        rechercher(args.requete, args.db)

if __name__ == "__main__":
    main()
