"""Module contenant la classe asbtraite QuestionReponsesFactorySparql
C'est cette classe qui doit être étendue pour créer des classes de fabrique de questions à choix multiples utilisant une entrée SPARQL du web des données.
"""
from abc import ABC, abstractmethod
from typing import Any
from rdflib import URIRef
from SPARQLWrapper import SPARQLWrapper, JSON
from .question_reponses import QuestionReponses


class QuestionReponsesFactorySparql(ABC):
    """Classe abstraite implémentant le protocole QuestionReponsesFactory pour les fabriques de questions à choix multiples utilisant une entrée SPARQL du web des données."""

    @abstractmethod
    def question_reponses(self) -> QuestionReponses:
        """Retourne une question à choix multiples utilisant une entrée SPARQL du web des données."""
        pass

    def prefixes(self) -> str:
        """Retourne les préfixes standards utiles pour faire une requête SPARQL.
        Cette méthode doit être étendue pour ajouter des préfixes spécifiques à une source de données.
        """
        return """PREFIX rdfs: <http://www.w3.org/2000/01/rdf-schema#>
PREFIX rdf: <http://www.w3.org/1999/02/22-rdf-syntax-ns#>
PREFIX xsd: <http://www.w3.org/2001/XMLSchema#>
"""

    def rdf_lang_string_en_str(self, rdf_lang_string: str) -> str:
        """Si lang_string est une chaîne de caractères de la forme "..."@XX, retourn "..."."""
        return rdf_lang_string.split("@")[0]

    @abstractmethod
    def uri_entree_sparql(self) -> URIRef:
        """Retourne l'entrée SPAQRL."""

    def execute_requete_sparql(self, la_requete: str, identifiants_sparql: list[str]) -> list[tuple[Any]]:
        """Exécute la requête SPARQL et retourne les résultats sous forme d'une liste de tuples.
        Chaque tuple contient les valeurs des variables de la requête.
        """
        sparql = SPARQLWrapper(self.uri_entree_sparql())
        sparql.setQuery(la_requete)
        sparql.setReturnFormat(JSON)
        resultats = sparql.query().convert()
        return [tuple([resultat[identifiant]["value"] for identifiant in identifiants_sparql]) for resultat in resultats["results"]["bindings"]]
