"""Programme principal pour poser des questions à choix multiples de culture générale."""
import importlib
import pkgutil
import random
from abc import ABC
from typing import Protocol
from question_reponses.question_reponses import QuestionReponses, QuestionReponsesFactory

class QuestionReponsesAleatoireFactory:
    """Fabrique de questions à choix multiples.
Cette fabrique choisit aléatoirement une fabrique de questions à choix multiples parmi celles disponibles.
Les fabriques disponibles, qui doivent se nommes QuestionReponsesFactory, sont découvertes automatiquement en explorant les modules du package 'question_reponses'
.
"""
    def __init__(self):
        self._les_fabriques = QuestionReponsesAleatoireFactory.decouvrir_et_instancier_fabriques("question_reponses")

    @staticmethod
    def decouvrir_et_instancier_fabriques(nom_package: str) -> list[QuestionReponsesFactory]:
        try:
            package = importlib.import_module(nom_package)
            les_fabriques = []
            for trouveur, nom_module, est_package in pkgutil.walk_packages(package.__path__,
                                                                           package.__name__ + "."):
                try:
                    module = importlib.import_module(nom_module)
                    if hasattr(module, "QuestionReponsesFactory"):
                        classe_fabrique = getattr(module, "QuestionReponsesFactory")
                        if not issubclass(classe_fabrique, Protocol):
                            print(f"Prise en compte du module {nom_module}")
                            les_fabriques.append(classe_fabrique())
                except (ImportError, AttributeError):
                    pass
            return les_fabriques
        except ImportError as e:
            raise ImportError(f"Le package '{nom_package}' n'a pas pu être importé.") from e

    def question_reponses(self):
        """Retourne une instance de QuestionReponses au hazard."""
        return random.choice(self._les_fabriques).question_reponses()

def poser_question(question_reponses: QuestionReponses) -> None:
    """Affiche la question et les réponses."""
    print(f"\nDomaine: {question_reponses.domaine}")
    print(f"Question: {question_reponses.question}")
    for i, reponse in enumerate(question_reponses.reponses):
        print(f"  Réponse {i + 1}: {reponse}")


def verifier_reponse(question_reponses: QuestionReponses) -> None:
    """Vérifie la réponse de l'utilisateur."""
    saisie_ok = False
    while not saisie_ok:
        try:
            reponse = int(input("Entrez le numéro de la réponse (-1 pour sortir): "))
            saisie_ok = reponse == -1 or 1 <= reponse <= len(question_reponses.reponses)
        except ValueError:
            pass
    if reponse == -1:
        exit()
    if reponse == question_reponses.indice_bonne_reponse + 1:
        print("Bonne réponse!")
    else:
        print("Mauvaise réponse!")

def main():
    fabrique_de_questions = QuestionReponsesAleatoireFactory()
    while (True):
        question_reponses = fabrique_de_questions.question_reponses()
        poser_question(question_reponses)
        verifier_reponse(question_reponses)

if __name__ == "__main__":
    main()
