#!/usr/bin/env python3
"""Module permettant de manipuler un plateau de Puissance 4"""

VIDE = 0
JAUNE = 1
ROUGE = 2

LARGEUR_PLATEAU = 7
HAUTEUR_PLATEAU = 6


def plateau_vide() -> "Plateau":
    """Fonction qui permet d'obtenir un plateau vide de LARGEUR_PLATEAU
    colonnes (1..LARGEUR_PLATEAU) chaque colonne étant de hauteur HAUTEUR_PLATEAU"""
    return [[VIDE] * LARGEUR_PLATEAU for ligne in range(HAUTEUR_PLATEAU)]

def jouer(plateau: "Plateau", colonne: int, pion: "JAUNE|ROUGE") -> None:
    """Fonction qui permet de jouer un pion dans un colonne pas encore
    remplie (hauteur_colonne(colonne) <= HAUTEUR_PLATEAU)"""
    plateau[hauteur_colonne(plateau, colonne)][colonne-1] = pion

def contenu_case(plateau: "Plateau", colonne: int, ligne:int) -> "Contenu":
    """Fonction qui permet d'obtenir le contenu d'une case du
    plateau: ROUGE, JANUE ou VIDE à partir de la colonne (1..LARGEUR_PLATEAU)
    et de la ligne (1..HAUTEUR_PLATEAU)"""
    return plateau[ligne-1][colonne-1]

def hauteur_colonne(plateau: "Plateau", colonne: int) -> int:
    """Fonction qui permet d'obtenir la hautreur d'une colonne"""
    ligne = HAUTEUR_PLATEAU
    while ligne > 0 and contenu_case(plateau, colonne, ligne) == VIDE:
        ligne -= 1
    return ligne

def colonne_remplie(plateau: "Plateau", colonne: int) -> bool:
    """Fonction qui permet de savoir si une colonne est remplie"""
    return hauteur_colonne(plateau, colonne) == HAUTEUR_PLATEAU

def plateau_rempli(plateau: "Plateau") -> bool:
    """Fonction qui permet de savoir si le plateau est rempli"""
    for colonne in range(1,8):
        if hauteur_colonne(plateau, colonne) < HAUTEUR_PLATEAU:
            return False
    return True

def plateau_est_vide(plateau: "Plateau") -> bool:
    """Fonction qui permet de savoir si un plateau est vide"""
    for colonne in range(1,8):
        if hauteur_colonne(plateau, colonne) > 0:
            return False
    return True

def nb_pions_alignes_horizontalement(plateau: "Plateau", colonne: int, ligne: int) -> int:
    """Fonction qui permet de savoir combien de pions d'une même
    couleur sont alignés horizontalement à patir d'une position donnée.
    Retourne 0 si à cette position la case est vide"""
    contenu = contenu_case(plateau, colonne, ligne)
    if contenu != VIDE:
        nb_a_gauche = 0
        i = colonne - 1
        while i > 0 and contenu_case(plateau, i, ligne) == contenu:
            nb_a_gauche += 1
            i -= 1
        nb_a_droite = 0
        i = colonne + 1
        while i <= LARGEUR_PLATEAU and contenu_case(plateau, i, ligne) == contenu:
            nb_a_droite += 1
            i += 1
        return 1 + nb_a_gauche + nb_a_droite
    return 0

def nb_pions_alignes_verticalement(plateau: "Plateau", colonne: int, ligne: int) -> int:
    """Fonction qui permet de savoir combien de pions d'une même
    couleur sont alignés verticalement à patir d'une position donnée.
    Retourne 0 si à cette position la case est vide"""
    contenu = contenu_case(plateau, colonne, ligne)
    if contenu != VIDE:
        nb_en_dessous = 0
        j = ligne - 1
        while j > 0 and contenu_case(plateau, colonne, j) == contenu:
            nb_en_dessous += 1
            j -= 1
        nb_au_dessus = 0
        j = ligne + 1
        while j <= HAUTEUR_PLATEAU and contenu_case(plateau, colonne, j) == contenu:
            nb_au_dessus += 1
            j += 1
        return 1 + nb_en_dessous + nb_au_dessus
    return 0

def nb_pions_alignes_diagonalement_SONE(plateau: "Plateau", colonne: int, ligne: int) -> int:
    """Fonction qui permet de savoir combien de pions d'une même
    couleur sont alignés diagonalement (du Sud-Ouest au Nord-Est)
    à patir d'une position donnée.
    Retourne 0 si à cette position la case est vide"""
    contenu = contenu_case(plateau, colonne, ligne)
    if contenu != VIDE:
        nb_SO = 0
        i = colonne -1
        j = ligne - 1
        while i > 0 and j > 0 and contenu_case(plateau, i, j) == contenu:
            nb_SO += 1
            i -= 1
            j -= 1
        nb_NE = 0
        i = colonne + 1
        j = ligne + 1
        while i <= LARGEUR_PLATEAU and j <= HAUTEUR_PLATEAU\
              and contenu_case(plateau, i, j) == contenu:
            nb_NE += 1
            i += 1
            j += 1
        return 1 + nb_SO + nb_NE
    return 0

def nb_pions_alignes_diagonalement_NOSE(plateau: "Plateau", colonne: int, ligne: int) -> int:
    """Fonction qui permet de savoir combien de pions d'une même
    couleur sont alignés diagonalement (du Nord-Ouest au Sud-Est)
    à patir d'une position donnée.
    Retourne 0 si à cette position la case est vide"""
    contenu = contenu_case(plateau, colonne, ligne)
    if contenu != VIDE:
        nb_NO = 0
        i = colonne -1
        j = ligne + 1
        while i > 0 and j <= HAUTEUR_PLATEAU and contenu_case(plateau, i, j) == contenu:
            nb_NO += 1
            i -= 1
            j += 1
        nb_SE = 0
        i = colonne + 1
        j = ligne - 1
        while i <= LARGEUR_PLATEAU and j > 0 and contenu_case(plateau, i, j) == contenu:
            nb_SE += 1
            i += 1
            j -= 1
        return 1 + nb_NO + nb_SE
    return 0

def tests():
    """Tests uniatires du modules"""
    plateau = plateau_vide()
    jouer(plateau,2,JAUNE)
    jouer(plateau,2,ROUGE)
    jouer(plateau,3,JAUNE)
    jouer(plateau,3,ROUGE)
    jouer(plateau,3,JAUNE)
    jouer(plateau,3,ROUGE)
    jouer(plateau,3,JAUNE)
    jouer(plateau,3,ROUGE)
    jouer(plateau,4,JAUNE)
    jouer(plateau,4,JAUNE)
    jouer(plateau,4,ROUGE)
    jouer(plateau,4,ROUGE)
    jouer(plateau,5,JAUNE)
    jouer(plateau,5,ROUGE)
    jouer(plateau,5,JAUNE)
    res = "OK" if contenu_case(plateau, 1,1) == VIDE else "KO"
    print(f"  contenu_case vide {res}")
    res = "OK" if contenu_case(plateau, 2,1) == JAUNE else "KO"
    print(f"  contenu_case JAUNE {res}")
    res = "OK" if contenu_case(plateau, 2,2) == ROUGE else "KO"
    print(f"  contenu_case ROUGE {res}")
    res = "OK" if hauteur_colonne(plateau, 1) == 0 else "KO"
    print(f"  hauteur_colonne vide {res}")
    res = "OK" if hauteur_colonne(plateau, 2) == 2 else "KO"
    print(f"  hauteur_colonne non vide {res}")
    res = "OK" if not colonne_remplie(plateau, 2) else "KO"
    print(f"  colonne_remplie non remplie {res}")
    res = "OK" if colonne_remplie(plateau, 3) else "KO"
    print(f"  colonne_remplie remplie {res}")
    res = "OK" if nb_pions_alignes_horizontalement(plateau, 3,1) == 4 else "KO"
    print(f"  nb_pions_alignes_horizontalement {res}")
    res = "OK" if nb_pions_alignes_verticalement(plateau, 4,1) == 2 else "KO"
    print(f"  nb_pions_alignes_verticalement {res}")
    res = "OK" if nb_pions_alignes_diagonalement_SONE(plateau, 4,3) == 2 else "KO"
    print(f"  nb_pions_alignes_diagonalement_SONE {res}")
    res = "OK" if nb_pions_alignes_diagonalement_NOSE(plateau, 4,3) == 3 else "KO"
    print(f"  nb_pions_alignes_diagonalement_NOSE {res}")

if __name__ == "__main__":
    tests()
